this.adapters = this.adapters || {};
(function () {
  'use strict';

  /**
   * Base class for SCORM adapters,basede on pipwerks.
   * @abstract
   * @class SCORMAdapter
   * @extends jsbeam.BaseAdapter
   * @memberof adapters
   * @alias adapters.SCORMAdapter
   * @classdesc Base classs for SCORM adapters
   * @constructor
   */
  function SCORMAdapter() {
    jsbeam.BaseAdapter.call(this);
    this._propertyController = null;
    this._scormCompletionValue = null;
    this._ignoreScore = false;
    this._version = '1.2';
  }

  SCORMAdapter.prototype = Object.create(jsbeam.BaseAdapter.prototype);
  SCORMAdapter.prototype.constructor = SCORMAdapter;

  /**
   * Init function of adpater that connects adapter to API, if not connected.
   * Sets values from SCORM server and inits objectives.
   * @public
   * @instance
   * @method connect
   * @memberof adapters.SCORMAdapter
   */
  SCORMAdapter.prototype.connect = function () {
    this._propertyController = miniinjector.Injector.injectById(jsbeam.JSBeamControllerID.PROPERTY_CONTROLLER);

    this._ignoreScore = !!this._propertyController.getPropertyFromCollection('config', 'preventScormScore');

    if (!this._isConnected) {

      jsbeam.BaseAdapter.prototype.connect.call(this);
      pipwerks.debug.isActive = this.debugMode;
      this._isConnected = pipwerks.SCORM.init();

      if (this._isConnected) {
        this._completed = this._getCourseCompleted();

        this._initData();

        this._objectivesCount = this._getObjectivesCount();

        this.dispatcher.dispatch(jsbeam.AdapterEvent.CONNECTED);

        this._scormCompletionValue = this._propertyController.getPropertyFromCollection('config', 'scormCompletionValue');
      } else {
        this.dispatcher.dispatch(jsbeam.AdapterEvent.FAILED_TO_CONNECT);
      }

    } else {
      return true;
    }
  };

  /**
   * Get course property for current course
   * @public
   * @override
   * @instance
   * @method getProperty
   * @memberof adapters.SCORMAdapter
   * @param  {string} id key of property
   * @return {string | null}    proeprty value or null
   */
  SCORMAdapter.prototype.getProperty = function (id) {

    if (this._isConnected) {
      return pipwerks.SCORM.get(id);
    }
    return null;
  };

  /**
   * Set course property for current course and commit to server
   * @public
   * @override
   * @instance
   * @method saveProperty
   * @memberof adapters.SCORMAdapter
   * @param  {string} id key of property
   * @param  {string} proeprty value
   * @return {boolean} success
   */
  SCORMAdapter.prototype.saveProperty = function (id, value) {

    if (this._isConnected) {
      var success = pipwerks.SCORM.set(id, value);

      if (success)
        pipwerks.SCORM.save();

      return success;
    }

    return false;
  };

  /**
   * Register objectives on server if there are none
   * @public
   * @override
   * @instance
   * @method initObjectives
   * @memberof adapters.SCORMAdapter
   * @param  {array} objectives objectives list
   */
  SCORMAdapter.prototype.initObjectives = function (objectives) {
    jsbeam.BaseAdapter.prototype.initObjectives.call(this, objectives);
    if (this._objectivesCount === 0) {
      this._registerObjectives(objectives);
    } else {
      this._updateObjectives(objectives);
    }

  };

  /**
   * Disconnect adapter from server, if connected
   * @public
   * @override
   * @instance
   * @method disconnect
   * @memberof adapters.SCORMAdapter
   */
  SCORMAdapter.prototype.disconnect = function () {
    if (this._isConnected) {
      jsbeam.BaseAdapter.prototype.disconnect.call(this);
      this._isConnected = false;
      pipwerks.SCORM.quit();
      this.dispatcher.dispatch(jsbeam.AdapterEvent.DISCONNECTED, this);
    }
  };

  adapters.SCORMAdapter = SCORMAdapter;
}());

this.adapters = this.adapters || {};
(function () {
  'use strict';

  /**
   * Implementation of SCORM 2004 adapter.
   * @class SCORMAdapter2004
   * @extends adapters.SCORMAdapter
   * @memberof adapters
   * @alias adapters.SCORMAdapter2004
   * @classdesc Implementation of SCORM 2004 adapte.
   * @constructor
   */
  function SCORMAdapter2004() {
    adapters.SCORMAdapter.call(this);
    pipwerks.SCORM.version = '2004';

    console.log('scormadapter init');
  }

  SCORMAdapter2004.prototype = Object.create(adapters.SCORMAdapter.prototype);
  SCORMAdapter2004.prototype.constructor = SCORMAdapter2004;

  /**
   * Get saved bookmark
   * @protected
   * @override
   * @instance
   * @method _getBookmark
   * @memberof adapters.SCORMAdapter2004
   * @return {string | null} bookmark
   */
  SCORMAdapter2004.prototype._getBookmark = function () {
    return this.getProperty('cmi.location');
  };

  /**
   * Save bookamrk
   * @public
   * @override
   * @instance
   * @method setBookmark
   * @memberof adapters.SCORMAdapter2004
   * @param  {string} site current bookamrk
   * @return {boolean} success
   */
  SCORMAdapter2004.prototype.setBookmark = function (site) {
    return this.saveProperty('cmi.location', site);
  };

  /**
   * Get user id from server
   * @protected
   * @override
   * @instance
   * @method _getUserID
   * @memberof adapters.SCORMAdapter2004
   * @return {string | null} user id
   */
  SCORMAdapter2004.prototype._getUserID = function () {
    return this.getProperty('cmi.learner_id');
  };

  /**
   * Get user name from server
   * @protected
   * @override
   * @instance
   * @method _getUserName
   * @memberof adapters.SCORMAdapter2004
   * @return {string|null} user name
   */
  SCORMAdapter2004.prototype._getUserName = function () {
    return this.getProperty('cmi.learner_name');
  };

  /**
   * Get total count of objectives
   * @protected
   * @override
   * @instance
   * @method _getObjectivesCount
   * @memberof adapters.SCORMAdapter2004
   * @return {number} count
   */
  SCORMAdapter2004.prototype._getObjectivesCount = function () {
    return coreutils.ExtendedMath.parseToInt(this.getProperty('cmi.objectives._count'));
  };

  /**
   * Check if the current course is completed
   * @protected
   * @override
   * @instance
   * @method _getCourseCompleted
   * @memberof adapters.SCORMAdapter2004
   * @return {boolean} completed
   */
  SCORMAdapter2004.prototype._getCourseCompleted = function () {
    var status = this.getProperty('cmi.completion_status');

    if (status === 'completed' || status === 'passed') {
      return true;
    }

    return false;
  };

  /**
   * Set objective in current course as completed
   * @public
   * @override
   * @instance
   * @method setObjectiveCompleted
   * @memberof adapters.SCORMAdapter2004
   * @param  {string} id    objective id
   * @return {boolean}      success
   */
  SCORMAdapter2004.prototype.setObjectiveCompleted = function (id) {
    var index = this._getObjectiveIndexByID(id);

    if (index !== null)
      this.saveProperty('cmi.objectives.' + index + '.completion_status', 'completed');
  };

  /**
   * Set objective in current course as incompleted
   * @public
   * @override
   * @instance
   * @method setObjectiveIncomplete
   * @memberof adapters.SCORMAdapter2004
   * @param  {string} id    objective id
   * @return {boolean}      success
   */
  SCORMAdapter2004.prototype.setObjectiveIncomplete = function (id) {
    var index = this._getObjectiveIndexByID(id);
    this.saveProperty('cmi.objectives.' + index + '.completion_status', 'incomplete');
  };

  /**
   * Register objectives on SCORM server
   * @protected
   * @override
   * @instance
   * @method _registerObjectives
   * @memberof adapters.SCORMAdapter2004
   * @param  {array} objectives
   */
  SCORMAdapter2004.prototype._registerObjectives = function (objectives) {

    var length = objectives.length;
    var trackedItem;

    for (var i = 0; i < length; i++) {
      trackedItem = objectives[i];

      if (!trackedItem.ignored) {
        this.saveProperty('cmi.objectives.' + this._getObjectiveIndexByID(trackedItem.id) + '.id', trackedItem.id);
        this.setObjectiveIncomplete(trackedItem.id);
      }
    }

  };

  /**
   * Check objective is completed and update controller
   * @protected
   * @instance
   * @method _updateObjectives
   * @memberof adapters.SCORMAdapter2004
   * @param  {array} objectives objectives list
   */
  SCORMAdapter2004.prototype._updateObjectives = function (objectives) {

    var length = objectives.length;
    var trackedItem;
    var status;

    for (var i = 0; i < length; i++) {
      trackedItem = objectives[i];

      if (!trackedItem.ignored) {
        status = this.getProperty('cmi.objectives.' + this._getObjectiveIndexByID(trackedItem.id) + '.completion_status');
        trackedItem.completed = (status == 'completed') ? true : false;
      }
    }

  };

  /**
   * Set current course status as completed
   * @public
   * @override
   * @instance
   * @method setCourseCompleted
   * @memberof adapters.SCORMAdapter2004
   */
  SCORMAdapter2004.prototype.setCourseCompleted = function () {
    if (!this._completed) {
      this._completed = true;
      this.saveProperty('cmi.completion_status', 'completed');
      this.saveProperty('cmi.success_status', 'passed');
    }
  };

  /**
   * Set score of the course
   * @public
   * @override
   * @instance
   * @method setScore
   * @memberof adapters.SCORMAdapter2004
   * @param  {number} score current score
   */
  SCORMAdapter2004.prototype.setScore = function (score) {
    if (this._ignoreScore) {
      return;
    }

    this.saveProperty('cmi.score.raw', score);

    console.log('raw score', score);
    console.log('this._minScore', this._minScore);
    console.log('this._maxScore', this._maxScore);

    // if (this._maxScore <= this._minScore) {
    //   return;
    // }

    this.saveProperty('cmi.score.min', this._minScore);
    this.saveProperty('cmi.score.max', this._maxScore);

    var progressRaw = (score - this._minScore) / (this._maxScore - this._minScore);
    var progressPercent = parseInt(progressRaw * 100);
    var progress = progressPercent / 100;
    console.log('progressRaw', progressRaw);
    console.log('progressPercent', progressPercent);

    console.log('cmi.progress_measure', progress);
    this.saveProperty('cmi.progress_measure', progress);

    console.log('cmi.score.scaled', progress);
    this.saveProperty('cmi.score.scaled', progress);

    // neu
    console.log('cmi.scaled_passing_score', progress);
    this.saveProperty('cmi.scaled_passing_score', progress);
  };

  /**
   * Set minimal score of the course
   * @public
   * @override
   * @instance
   * @method setMinScore
   * @memberof adapters.SCORMAdapter2004
   * @param  {number} score minimal score
   */
  SCORMAdapter2004.prototype.setMinScore = function (score) {
    if (this._ignoreScore) {
      return;
    }
    adapters.SCORMAdapter.prototype.setMinScore.call(this, score);
    this.saveProperty('cmi.score.min', score);
  };

  /**
   * Set maximal score of the course
   * @public
   * @override
   * @instance
   * @method setMaxScore
   * @memberof adapters.SCORMAdapter2004
   * @param  {number} score maximal score
   */
  SCORMAdapter2004.prototype.setMaxScore = function (score) {
    if (this._ignoreScore) {
      return;
    }
    adapters.SCORMAdapter.prototype.setMaxScore.call(this, score);
    this.saveProperty('cmi.score.max', score);
  };

  SCORMAdapter2004.prototype.setSuspendData = function (data) {
    this.saveProperty('cmi.suspend_data', data);
  };

  SCORMAdapter2004.prototype.getSuspendData = function () {
    return this.getProperty('cmi.suspend_data');
  };

  adapters.SCORMAdapter2004 = SCORMAdapter2004;
}());

//# sourceMappingURL=Scorm_2004_Adapter.js.map